import { ethers as hardhatEthers } from 'hardhat';
import { SignerWithAddress } from '@nomiclabs/hardhat-ethers/signers';
import '@openzeppelin/hardhat-upgrades';
import { HttpStatus, INestApplication } from '@nestjs/common';
import request from 'supertest';
import {
  getPrivateApp,
  getEthRpcProvider,
  getPublicApp,
  // offeringsArr,
  getSignerWallet,
} from '../utils/setup';
import { getContractAddresses } from '@contracts';
import { ethers } from 'ethers';
import { getIDHashHelper } from '../../../../libs/shared-utils/src/helpers/token-id.utils';

let signerWithAddress: SignerWithAddress[];
let privateApp: INestApplication;
let publicApp: INestApplication;
let user1: ethers.Wallet;
let provider1: ethers.providers.JsonRpcProvider;
let contracts: any;
const durationOneYear: string = '31536000'; // one year in seconds, 365*24*60*60
const durationTimeSec: number = parseInt(durationOneYear);
const unixTimeSec = Math.floor(Date.now() / 1000);

describe('Submission order user flow(e2e)', () => {
  beforeAll(async () => {
    privateApp = await getPrivateApp();
    await privateApp.init();

    publicApp = await getPublicApp();
    await publicApp.init();

    signerWithAddress = await hardhatEthers.getSigners();
    user1 = await getSignerWallet();
    provider1 = await getEthRpcProvider();
  });

  it(`Create offer`, async () => {
    const response = await request(privateApp.getHttpServer())
      .post('/tm/v1.0/offerings')
      .send({
        assetid: 'asub1',
        oid: 'osub1',
        resource: 'http://resource1.jpg',
        beneficiary: '0x57fFC30a883932F98fE94Fc0eE4390C4BA5A9F2a',
        price: '10',
        cap_duration: '31536000000',
        cds_target: { key: 'value' },
        cds_sl: { key: 'value' },
      });
    expect(response).toBeTruthy();
    expect(response.statusCode).not.toEqual(HttpStatus.NOT_FOUND);
  });

  it(`Mint payment tokens for user1`, async () => {
    const response = await request(privateApp.getHttpServer())
      .post('/tm/v1.0/payment-tokens/mint')
      .send({
        tradingAccount: signerWithAddress[1].address,
        amount: '1000',
      });

    expect(response).toBeTruthy();
    expect(response.statusCode).toBe(HttpStatus.CREATED);
  });

  it(`Approve operator to spent payment tokens from user1 account`, async () => {
    const response = await request(publicApp.getHttpServer())
      .get('/payment/approve-operator-to-spend-payment-tokens')
      .expect(200);

    const rawTransaction = response.body;

    contracts = getContractAddresses(process.env.HARDHAT_NETWORK);

    expect(rawTransaction).toBeTruthy();
    expect(rawTransaction).toHaveProperty('to');
    expect(hardhatEthers.utils.isAddress(rawTransaction.to)).toBeTruthy();
    expect(rawTransaction.to).toEqual(
      `${contracts.contractAddressPaymentToken}`,
    );

    const tx = {
      ...rawTransaction,
    };

    const estimatedGasLimit = await user1.estimateGas(tx);

    const gasPrice = await provider1.getGasPrice();

    tx.gasLimit = estimatedGasLimit;
    tx.gasPrice = gasPrice;

    //send approval
    const txResponse = await user1.sendTransaction(tx);
    expect(txResponse).toBeTruthy();
  });

  it(`Purchase asset`, async () => {
    const response = await request(publicApp.getHttpServer())
      .post('/submissions/order')
      .send({
        oid: 'osub1',
      });
    // console.log(response);
    expect(response).toBeTruthy();
    expect(response.statusCode).toBe(HttpStatus.CREATED);

    const transactionContext = JSON.parse(response.text);

    const tx: any = {
      ...transactionContext,
      from: user1.address,
    };

    const estimatedGasLimit = await provider1.estimateGas(tx);
    const gasPrice = await provider1.getGasPrice();

    tx.gasLimit = estimatedGasLimit;
    tx.gasPrice = gasPrice;

    await user1.sendTransaction(tx);

    // expect(typeof txResponse).toBe('string');

    // get assetid
    const { text: assetId } = await request(publicApp.getHttpServer())
      .get('/data-access-info/get-tokenid-from-offerid')
      .query({
        offerId: 'osub1',
      });

    console.log({ assetId });

    expect(assetId.length).toBeGreaterThanOrEqual(66);
    // check if asset was transfered
    const checkClearance = await request(publicApp.getHttpServer())
      .get('/data-access-info/check-clearance')
      .query({
        assetId: assetId,
        addresses: [user1.address],
        tokenType: 'SUB',
      });

    // check if asset was transfered
    expect(checkClearance.text).toBe('true');
  });

  it(`Access right expiration time is returned for purchased subscription`, async () => {
    const response = await request(publicApp.getHttpServer())
      .get('/data-access-info/subscription-last-expiration')
      .query({
        userAddress: user1.address,
        tokenId: getIDHashHelper('osub1'),
      });

    expect(response).toBeTruthy();
    expect(response.statusCode).toBe(HttpStatus.OK);
    expect(typeof response.text).toBe('string');
    expect(/^[0-9]+$/.test(response.text)).toBeTruthy();
    const expirationTime = parseInt(response.text, 10);
    expect(expirationTime).toBeGreaterThanOrEqual(
      unixTimeSec + durationTimeSec,
    );
  });

  it(`Zero is returned as access right expiration time of non-existent subscription`, async () => {
    const response = await request(publicApp.getHttpServer())
      .get('/data-access-info/subscription-last-expiration')
      .query({
        userAddress: user1.address,
        tokenId: getIDHashHelper('osub1nonexistent'),
      });

    expect(response).toBeTruthy();
    expect(response.statusCode).toBe(HttpStatus.OK);
    expect(typeof response.text).toBe('string');
    expect(/^[0-9]+$/.test(response.text)).toBeTruthy();
    const expirationTime = parseInt(response.text, 10);
    expect(expirationTime).toEqual(0);
  });

  it(`Trading history contains purchased asset`, async () => {
    const response = await request(publicApp.getHttpServer())
      .get('/trading-history')
      .query({
        oid: 'osub1',
      });

    expect(response).toBeTruthy();
    expect(response.statusCode).toBe(HttpStatus.OK);
    const respJson = JSON.parse(response.text);
    expect(respJson).toBeTruthy();
    expect(respJson['osub1']).toBeTruthy();
  });

  it('Should return default sales statistics when no body params are provided', async () => {
    const response = await request(publicApp.getHttpServer())
      .get('/trading-history/offering-sales-statistics')
      .send({ intervalUnits: 'block', stats: 'assets' });

    expect(response.status).toBe(200);
    expect(Array.isArray(response.body)).toBe(true);
    expect(response.body.length).toBeGreaterThan(0);
  });

  it('Should return sales statistics with intervalLength and intervalEnd', async () => {
    const response = await request(publicApp.getHttpServer())
      .get('/trading-history/offering-sales-statistics')
      .send({
        intervalLength: '10',
        intervalEnd: '100000',
        intervalUnits: 'block',
        stats: 'assets',
      });

    expect(response.status).toBe(200);
    expect(Array.isArray(response.body)).toBe(true);
  });

  it('Should return sales statistics filtered by intervalUnits as minute', async () => {
    const response = await request(publicApp.getHttpServer())
      .get('/trading-history/offering-sales-statistics')
      .send({ intervalUnits: 'minute', stats: 'assets' });

    expect(response.status).toBe(200);
    expect(Array.isArray(response.body)).toBe(true);
  });

  it('Should return sales statistics grouped by offerings', async () => {
    const response = await request(publicApp.getHttpServer())
      .get('/trading-history/offering-sales-statistics')
      .send({ intervalUnits: 'block', stats: 'offerings' });

    expect(response.status).toBe(200);
    expect(Array.isArray(response.body)).toBe(true);
    expect(Array.isArray(response.body[0])).toBe(true);

    // Check for the specific offering sale
    const offeringExists = response.body.some(
      (offering) => offering[0] === 'asub1' && offering[1] === 'osub1',
    );
    expect(offeringExists).toBe(true);
  });

  it('Should return transactions with correct structure and values', async () => {
    const response = await request(publicApp.getHttpServer())
      .get('/trading-history/offering-sales-statistics')
      .send({ intervalUnits: 'block', stats: 'transactions' });

    expect(response.status).toBe(200);
    expect(Array.isArray(response.body)).toBe(true);

    response.body.forEach((transaction) => {
      // Ensure 6 elements: [assetid, oid, price, payment token, token amount, unixtime]
      expect(transaction.length).toBe(6);

      const [assetid, oid, price, paymentToken, tokenAmount, unixtime] =
        transaction;

      expect(typeof assetid).toBe('string');
      expect(typeof oid).toBe('string');
      expect(typeof price).toBe('number');
      expect(typeof paymentToken).toBe('string');
      expect(typeof tokenAmount).toBe('number');
      expect(typeof unixtime).toBe('number');

      // Check for specific values
      if (assetid === 'asub1' && oid === 'osub1') {
        expect(paymentToken).toBe('FDE');
        expect(tokenAmount).toBe(1);
      }
    });
  });
});
